# NGArgumentParser

This is a standardized parser framework for CLI tools. This class will enforce certain properties or abstract methods to be implemented to properly create an Argument Parser class for other CLI tools.

## What's in NGArgumentParser?
The parser class inherits directly from `argparse.ArgumentParser`. Once it's instantiated, it will create an ArgumentParser class, and create 3 subparsers: `predict`, `preprocess`, and `postprocess`.

The `preprocess` subparser has the following options available:
```bash
>> python run_test.py preprocess -h
usage: run_test.py preprocess [-h] [--json-input JSON_FILE] [--dir SPLIT_PARAMETERS_DIR] [--inputs-dir SPLIT_INPUTS_DIR]

This is where it will split the jobs and output job_descriptions.json.

options:
  -h, --help            show this help message and exit
  --json-input JSON_FILE, -j JSON_FILE
                        JSON file containing input parameters.
  --dir SPLIT_PARAMETERS_DIR
                        the diretory for the JSON files that input parameters splitted into
  --inputs-dir SPLIT_INPUTS_DIR
                        the diretory for the sequence and peptide files that input sequences splitted into
```

The `postprocess` subparser has the following options available:
```bash
>> python run_test.py postprocess -h
usage: run_test.py postprocess [-h] [--job-desc-file JOB_DESC_FILE] [--aggregate-input-dir AGGREGATE_INPUT_DIR] [--aggregate-result-dir AGGREGATE_RESULT_DIR]

This is where it will aggregate the results into a file.

options:
  -h, --help            show this help message and exit
  --job-desc-file JOB_DESC_FILE
                        the file path for the job description
  --aggregate-input-dir AGGREGATE_INPUT_DIR
                        the diretory for the JSON files which have input parameters
  --aggregate-result-dir AGGREGATE_RESULT_DIR
                        the diretory for the JSON files contains results need to be aggregated as well as the place we place the final result file
```

The `predict` subparser has no options available as the NGArgumentParser will only create the subparser and not add any options to it. It is the developer's duty to add arguments and customize the `predict` subparser.

Let's now try to create a child class that builds off of this parser framework.


> __NOTE__ :
> 
> These are subject to change in the future.


## Create a child class
Let's create an example class that extends from this parser class.
```python
class ExampleArgumentParser(NGArgumentParser):
    def __init__(self):
        super().__init__()
```

To fully create an Argument Parser class using `NGArgumentParser`, a couple of elements must be implemented:

1. customizing program's detail (e.g. `prog`, `usage`, `description`, `epilog`)
2. add subparser for prediction
3. add tool-specific parameters to the prediction subparser

### 1. Customizing program's detail
```python
class ExampleArgumentParser(NGArgumentParser):
    def __init__(self):
        super().__init__()

        # Write your description HERE
        self.description="This is an example description code."
```
The `self.description` will update the ArgumentParser's description from NGArgumentParser.
```python
class NGArgumentParser(argparse.ArgumentParser):
    def __init__(self):
        '''
        It is the developer's responsibility to customize these parameters.
        At the minimum, the below parameters should be customized before deploying.

        Developers can choose to futher customize other parameters of ArgumentParser()
        from here:
        https://docs.python.org/3/library/argparse.html#argparse.ArgumentParser
        '''
        super().__init__()
        # self.prog='The name of the program (default: os.path.basename(sys.argv[0]))'
        # self.usage='The string describing the program usage (default: generated from arguments added to parser)'
        self.description='Text to display before the argument help (by default, no text)'
        self.epilog='Text to display after the argument help (by default, no text)'
```
By doing so, when `--help` is passed, it will change the program's description.
```bash
>> python run_test.py --help
usage: run_test.py [-h] {split,aggregate,predict} ...

This is an example description.

options:
  -h, --help            show this help message and exit

...
```
Please refer to the [argparse document](https://docs.python.org/3/library/argparse.html#argparse.ArgumentParser) to learn about other parameters.


### 2. Add subparser for prediction
The `NGArgumentParser` already handles creating the `subparser` inside the `add_predict_subparser()`.
```python
class ExampleArgumentParser(NXGArgumentParserBuilder):
    def __init__(self):
        .
        .
        pred_parser = self.add_predict_subparser(
            help='Perform individual prediction.',
            description='This is where the users can perform indifidual predictions.'
        )
```
The `add_predict_subparser()` will return a `subparser`.

### 3. Add tool-specific parameters to the prediction subparser
Since we can access prediction subparser from the child class, we can add any tool-specific commands to it.

```python
class ExampleArgumentParser(NXGArgumentParserBuilder):
    def __init__(self):
        .
        .
        pred_parser = self.add_predict_subparser(
            help='Perform individual prediction.',
            description='This is where the users can perform indifidual predictions.'
        )

        # Add tool-specific params 
        # -----------------------------------------------------
        pred_parser.add_argument("--output-prefix", "-o",
                                 dest="output_prefix",
                                 help="prediction result output prefix.",
                                 metavar="OUTPUT_PREFIX")
        pred_parser.add_argument("--output-format", "-f",
                                 dest="output_format",
                                 default="tsv",
                                 help="prediction result output format (Default=tsv)",
                                 metavar="OUTPUT_FORMAT")  
```

## Usage
In the executable file, the child class should be instantiated and logic to handle the cli-tool should be there. Let's create the executable file called `run_test.py`.

```python
def main():
    arg_parser = ExampleArgumentParser()
    args = arg_parser.parse_args()

    if args.subcommand == 'predict':
        ########################################################################
        # PREDICTION LOGIC GOES HERE.
        # 
        # 
        ########################################################################
    
    if args.subcommand == 'preprocess':
        # Retrieve attributes
        json_filename = getattr(args, 'json_filename')
        split_parameters_dir = getattr(args, 'split_parameters_dir')
        split_inputs_dir = getattr(args, 'split_inputs_dir')
        assume_valid_flag = getattr(args, 'assume_valid_flag')

        ########################################################################
        # SPLIT FUNCTION FROM "SPLIT.PY" SHOULD GO HERE.
        # 
        # 
        ########################################################################
    
    if args.subcommand == 'postprocess':       
        # Retrieve attributes 
        job_desc_file = getattr(args, 'job_desc_file')
        aggregate_input_dir = getattr(args, 'aggregate_input_dir')
        aggregate_result_dir = getattr(args, 'aggregate_result_dir')

        ########################################################################
        # AGGREGATE FUNCTION FROM "AGGREGATE.PY" SHOULD GO HERE.
        # 
        # 
        ########################################################################

if __name__=='__main__':
    main()
```
