import unittest
import os
import pandas as pd
import json
from nxg_common import set_python_env, api_results_json_to_df

class TestPythonEnv(unittest.TestCase):
    '''Basic unit test for nxg common'''
    def test_no_args(self):
        self.assertEqual(set_python_env(), [])
    
    def test_modules_only(self):
        self.assertEqual(set_python_env(modules="numpy,scipy"), ["module load numpy", "module load scipy"])
    
    def test_virtualenv_only(self):
        os.mkdir('test_venv')
        self.assertEqual(set_python_env(virtualenv='test_venv'), ["source test_venv/bin/activate"])
        os.rmdir('test_venv')
    
    def test_modules_and_virtualenv(self):
        os.mkdir('test_venv')
        self.assertEqual(set_python_env(modules="numpy", virtualenv='test_venv'), ["module load numpy", "source test_venv/bin/activate"])
        os.rmdir('test_venv')


class TestApiResultsJsonToDf(unittest.TestCase):
    @classmethod
    def setUpClass(cls):
        """Create a mock JSON file for testing."""
        cls.test_json_file = "test_data.json"
        cls.test_data = {
            "data": {
                "results": [
                    {
                        "type": "table1",
                        "table_columns": [
                            {"name": "col1"},
                            {"name": "col2"}
                        ],
                        "table_data": [
                            [1, "A"],
                            [2, "B"]
                        ]
                    },
                    {
                        "type": "table2",
                        "table_columns": [
                            {"name": "colA"},
                            {"name": "colB"}
                        ],
                        "table_data": [
                            [10, "X"],
                            [20, "Y"]
                        ]
                    }
                ]
            }
        }
        with open(cls.test_json_file, "w") as f:
            json.dump(cls.test_data, f)

    @classmethod
    def tearDownClass(cls):
        """Clean up the mock JSON file after testing."""
        os.remove(cls.test_json_file)

    def test_api_results_json_to_df(self):
        """Test the function with a valid JSON file and table types."""
        # Define the table types to extract
        table_types = ["table1"]

        # Call the function
        result = api_results_json_to_df(self.test_json_file, table_types)

        # Verify the result
        self.assertIsInstance(result, dict)  # Check if the result is a dictionary
        self.assertIn("table1", result)  # Check if the expected table type is in the result
        self.assertIsInstance(result["table1"], pd.DataFrame)  # Check if the value is a DataFrame

        # Verify the DataFrame content
        expected_df = pd.DataFrame(
            [[1, "A"], [2, "B"]],
            columns=["col1", "col2"]
        )
        pd.testing.assert_frame_equal(result["table1"], expected_df)

    def test_api_results_json_to_df_no_matching_tables(self):
        """Test the function with table types that do not exist in the JSON."""
        # Define table types that do not exist in the JSON
        table_types = ["non_existent_table"]

        # Call the function
        result = api_results_json_to_df(self.test_json_file, table_types)

        # Verify the result
        self.assertEqual(result, {})  # Check if the result is an empty dictionary

if __name__ == "__main__":
    unittest.main()