#!/usr/bin/env python

import os
from copy import deepcopy

# The settings, keyed by environment
all_settings = {
    'prod': {
        # global settings for the prod environment
        'global': {
            # slurm API token
            'token': os.environ.get('NXG_AUTH_TOKEN'),
            # URI of the API
            'base_uri': 'https://api-nextgen-tools.iedb.org/api/v1',
            # toplevel directory to store stage data
            'scratch_dir': '/scratch/stages',
            # modules & virtualenv to be loaded for running scripts in the nxg-tools package
            'modules_nxgtools': 'python/3.11.3',
            'virtualenv_nxgtools': '/apps/python-virtualenvs/nxg-tools'
        },
        # settings for the individual tool groups in the prod environment
        'tool_group': {
            'mhci': {
                'cmdline_path': '/apps/iedbtools/ng_tc1/src/tcell_mhci.py',
                'mem_mb': 5000,
                'modules': 'python/3.11.3',
                'virtualenv': '/apps/python-virtualenvs/tcell_mhci'
            },
            'mhcii': {
                'cmdline_path': '/apps/iedbtools/ng_tc2/src/tcell_mhcii.py',
                'mem_mb': 2000,
                'modules': 'python/3.11.3',
                'virtualenv': '/apps/python-virtualenvs/tcell_mhci'
            },
            'icerfire': {
                'cmdline_path': '/apps/iedbtools/ng_pvc/src/run_pvc.py',
                'mem_mb': 2000,
                'modules': 'python/3.8.18',
                'virtualenv': '/apps/python-virtualenvs/icerfire'
            },
            'peptide_variant_comparison': {
                'cmdline_path': '/apps/iedbtools/ng_pvc/src/run_pvc.py',
                'mem_mb': 500,
                'modules': 'python/3.11.3',
                'virtualenv': '/apps/python-virtualenvs/tcell_mhci'
            },
            'mutgen': {
                'cmdline_path': '/apps/iedbtools/ng_mpg/src/run_mutgen.py',
                'mem_mb': 2500,
                'modules': 'python/3.11.3',
                'virtualenv': '/apps/python-virtualenvs/mpg'
            },
            'cluster': {
                'cmdline_path': '/apps/iedbtools/ng_cluster/src/run_cluster.py',
                'mem_mb': 500,
                'modules': 'python/3.8.18',
                'virtualenv': '/apps/python-virtualenvs/cluster-analysis'
            },
            'pepmatch': {
                'cmdline_path': '/apps/iedbtools/ng_pepmatch/src/match.py',
                'mem_mb': 5000,
                'env_vars': {
                    'PEPMATCH_PROTEOMES_PATH': '/apps/iedbtools/pepmatch-proteomes'
                            },
                'modules': 'python/3.11.3',
                'virtualenv': '/apps/python-virtualenvs/pepmatch'
            }
        }
    },
    'dev': {
        # global settings for the dev environment
        'global': {
            # slurm API token
            'token': os.environ.get('NXG_AUTH_TOKEN'),
            # URI of the API
            'base_uri': 'https://api-nextgen-tools-dev.iedb.org/api/v1',
            # toplevel directory to store stage data
            'scratch_dir': '/scratch/stages',
            # modules & virtualenv to be loaded for running scripts in the nxg-tools package
            'modules_nxgtools': 'python/3.11.3',
            'virtualenv_nxgtools': '/apps/python-virtualenvs/nxg-tools'
        },
        # settings for the individual tool groups in the prod environment
        'tool_group': {
            'mhci': {
                'cmdline_path': '/apps/iedbtools/ng_tc1/src/tcell_mhci.py',
                'mem_mb': 2000,
                'modules': 'python/3.11.3',
                'virtualenv': '/apps/python-virtualenvs/tcell_mhci'
            },
            'mhcii': {
                'cmdline_path': '/apps/iedbtools/ng_tc2/src/tcell_mhcii.py',
                'mem_mb': 2000,
                'modules': 'python/3.11.3',
                'virtualenv': '/apps/python-virtualenvs/tcell_mhci'
            },
            'peptide_variant_comparison': {
                'cmdline_path': '/apps/iedbtools/ng_pvc/src/run_pvc.py',
                'mem_mb': 500,
                'modules': 'python/3.11.3',
                'virtualenv': '/apps/python-virtualenvs/tcell_mhci'
            },
            'icerfire': {
                'cmdline_path': '/apps/iedbtools/ng_pvc/src/run_pvc.py',
                'mem_mb': 2000,
                'modules': 'python/3.8.18',
                'virtualenv': '/apps/python-virtualenvs/icerfire'
            },
            'mutgen': {
                'cmdline_path': '/apps/iedbtools/mutated-peptide-generator/src/run_mutgen.py',
                'mem_mb': 500,
                'modules': 'python/3.11.3',
                'virtualenv': '/apps/python-virtualenvs/tcell_mhci'
            },
            'cluster': {
                'cmdline_path': '/apps/iedbtools/ng_cluster-0.1/src/run_cluster.py',
                'mem_mb': 500,
                'modules': 'python/3.8.18',
                'virtualenv': '/apps/python-virtualenvs/cluster-analysis'
            },
            'pepmatch': {
                'cmdline_path': '/apps/iedbtools/ng_pepmatch-0.1/src/match.py',
                'mem_mb': 5000,
                'env_vars': {
                    'PEPMATCH_PROTEOMES_PATH': '/apps/iedbtools/pepmatch-proteomes'
                            },
                'modules': 'python/3.11.3',
                'virtualenv': '/apps/python-virtualenvs/pepmatch'
            }
        }
    }
}


def load_global_settings(env_name, passed_args):
    """Given an environment name (prod, dev, local),
    as well as the arguments passed to the commane line tool return
    the proper settings"""
        
    if env_name not in all_settings:
        #TODO: die here
        print(f"Environment name {env_name} is not found in the settings file")
        all_names = all_settings.keys()
        print(f"Possible values: {all_names}")
        exit(1)

    # now let's return the proper settings undereath the enviroment
    settings = deepcopy(all_settings[env_name]['global'])

    # if any of the settings are overriden by command line settings, let's
    # take care of that here
    for arg in vars(passed_args):
        arg_value = getattr(passed_args, arg)
        if arg_value:
            # if we are overriding a value that was in the settins file, report it here
            if arg in settings:
                print(f"Overriding value from settings file for {arg} ({settings[arg]}) with value from command line: {arg_value}")
                
            settings[arg] = arg_value
        
    return settings


def load_tool_group_settings(env_name, tool_group, passed_args=None):
    """Given an environment name (prod, dev, local, etc.),
    tool_group name (mhci, cluster, pepmatch, etc.,)
    as well as the arguments passed to the commane line tool return
    the proper settings"""
    
    if env_name not in all_settings:
        #TODO: die here
        print(f"Environment name {env_name} is not found in the settings file")
        all_names = all_settings.keys()
        print(f"Possible values: {all_names}")
        exit(1)
    
    if tool_group not in all_settings[env_name]['tool_group']:
         #TODO: die here
        print(f"Tool group name {tool_group} is not found in the settings file for this environment")
        all_names = all_settings[env_name].keys()
        print(f"Possible values: {all_names}")
        exit(1)       

    # now let's return the proper settings undereath the enviroment
    tool_settings = deepcopy(all_settings[env_name]['tool_group'][tool_group])

    # if any of the settings are overriden by command line settings, let's
    # take care of that here
    if passed_args:
        common_args = set(vars(passed_args)) & set(tool_settings.keys()) 
        for arg in common_args:
            arg_value = getattr(passed_args, arg)
            if arg_value:
                print(f"Overriding value from settings file for {arg} ({tool_settings[arg]}) with value from command line: {arg_value}")
                tool_settings[arg] = arg_value

    return tool_settings
