#! /usr/bin/env python3

import os
import importlib.util

CONFIG_PATH = "paths.py"
ENV_INFO_PATH = ".env_info"

DEP_TOOL_MAPPER = {
    "mhci": "tcell_class_i",
    "mhcii": "tcell_class_ii"
}

def load_config(path):
    if not os.path.exists(path):
        print(f"❌ Config file '{path}' not found.")
        return {}
    
    # Treat 'path' as a module and load everything into 'config'
    spec = importlib.util.spec_from_file_location("config", path)
    config = importlib.util.module_from_spec(spec)
    spec.loader.exec_module(config)

    return {
        k: v for k, v in config.__dict__.items()
        if not k.startswith("__")
    }

def write_env_info(config, output_path):
    with open(output_path, "w") as f:
        for key, value in config.items():
            if value is None:
                continue

            if isinstance(value, str):
                value = value.strip("'").strip('"')
            
            f.write(f"{key.upper()}={value}\n")

    # print(f"* .env_info file created at '{output_path}'")
    print(f"* .env_info file created")

def create_shell_script(config, tool, output_path):
    lines = ["#!/bin/bash\n"]

    # Define keys based on the tool name
    prefix = DEP_TOOL_MAPPER.get(tool)

    if not prefix:
        raise ValueError(f"Unknown tool '{tool}'")

    # Get values from config using prefix
    module = config.get(f"{prefix}_module")
    venv = config.get(f"{prefix}_venv")
    lib_path = config.get(f"{prefix}_lib_path")
    tool_path = config.get(f"{prefix}_path")
    env_var = f"{prefix.upper()}_PATH"

    lines.append(f"# ---- Setup for {tool.upper()} ----")

    # Optional: Load module
    if module:
        lines.append("module purge")
        lines.append(f"module load {module}")

    # Optional: Activate virtualenv
    if venv:
        lines.append(f"source {venv}/bin/activate")

    # Optional: Set LD_LIBRARY_PATH
    if lib_path:
        lines.append(f"export LD_LIBRARY_PATH={lib_path}:$LD_LIBRARY_PATH")

    # Required: Export tool path
    if tool_path:
        lines.append(f"export {env_var}={tool_path}")

    with open(output_path, "w") as f:
        f.write("\n".join(lines) + "\n")

    os.chmod(output_path, 0o755)
    print(f"* Shell script for '{tool}' created at '{output_path}'")

def main():
    config = load_config(CONFIG_PATH)
    if not config:
        return
    write_env_info(config, ENV_INFO_PATH)

    for tool in DEP_TOOL_MAPPER.keys():
        create_shell_script(config, tool, output_path=f'setup_{tool}_env.sh')


if __name__ == "__main__":
    main()
