# Functions that may be generally useful to multiple scripts

import os
import tempfile
import json
import pandas as pd
from urllib.request import urlopen
from shutil import copyfileobj

def set_python_env(modules: str=None, virtualenv: str=None):    
    """
    From the pythonpath, virtualenv, & module command line arguments,
    determine the list of commands that needs to be run to set the
    python environment
    """
    
    python_env_cmds = list()
    
    if modules:
        modules_list = [s.strip() for s in modules.split(",")]
        for m in modules_list:
            # if the module is separated by path, we must try to unload
            # the base path
            module_parts = m.split('/')
            if len(module_parts) > 1:
                python_env_cmds.append("module unload " + module_parts[0])
            python_env_cmds.append("module load " + m)

    if virtualenv:
        activate_path = os.path.join(virtualenv,'bin', 'activate')
        python_env_cmds.append("source " + activate_path)
        
    return python_env_cmds


def get_fully_qualified_field_name(field_dict):
    """Given a field descriptor dictionary, return the fully-qualified field name"""
    # if we still use 'binding' in the field name if the source == 'binding/elution'
    return field_dict['source'].replace('binding/elution','binding') + '.' + field_dict['name']


# Likely there is a python funciton that does this already...
def catalog_list_by(input_list, field_name):
    """Given a list of dicts, return a dictionary that maps
    the field name to the index of the list"""
    
    list_map = dict()
    
    for i in range(len(input_list)):
        # TODO: check that the value exists
        list_map[input_list[i][field_name]] = i
    
    return list_map

# read a JSON file
def read_json_file(json_file):
    """Given a JSON file, read it in and 
    return the corresponding data
    """
    
    with open(json_file, 'r') as json_data:
        data = json.load(json_data)

    return data


# save file from URI
def save_file_from_URI(file_uri, output_file=None, target_dir=None):
    """Given the URI of a file and optionally an output file name or target directory,
    Retreive the file from the URI and save it.  Return the full path to the file"""
    
    # first define the output file and/or directory
    if output_file == None:
        if target_dir == None:
            # set the target dir to the temp directory
            target_dir = tempfile.gettempdir()
    
        output_file = tempfile.NamedTemporaryFile(dir=target_dir, delete=False)
    
    #print("Copying from: " + file_uri)
    #print("Copying to: " + output_file.name)
    
    with urlopen(file_uri) as fsrc, output_file as fdst:
        copyfileobj(fsrc, fdst)
        return fdst.name
    
    
def api_results_json_to_df(json_file, table_types=[]):
    """ Given a JSON file with results data and the names of the table(s)
    to extract, return a dict of pandas dataframes with the data, keyed by
    the table type name
    """
    
    # read in the JSON file
    with open(json_file, 'r') as f:
        data = json.load(f)

    # Filter tables where the 'type' field matches one of the target types
    filtered_tables = [table for table in data["data"]["results"] if table.get("type") in table_types]

    # Process and convert each matching table to a DataFrame
    dfs = dict()
    for table in filtered_tables:
        columns = [col["name"] for col in table["table_columns"]]
        rows = table["table_data"]
        df = pd.DataFrame(rows, columns=columns)
        dfs[table["type"]] = df  # Store DataFrame with its type name

    # Display results
    if dfs:
        for table_type, df in dfs.items():
            print(f"\nTable Type: {table_type}")
            print(df.head())  # Display first few rows
    else:
        all_types = ", ".join(table_types)
        print(f"Error: No tables matching requested types ({all_types}) found.")
        
    return dfs