import argparse
import os
import sys
from pathlib import Path
sys.tracebacklimit = 0
METHOD_DIR = os.path.join(os.getcwd(), 'method')
sys.path.insert(0, str(METHOD_DIR))
from allele_info import MHCIAlleleData
from axelf_executable import Axelf


USAGE_INFO = "" \
            "usage: axelf.py [-h] [-p] inputFile -s [TPM_SOURCE] [-a " \
            "[ALLELE_LIST]] [-c [CANCER_TYPE]] [-l [PEPTIDE_LENGTH]] " \
            "[-g [GENE_NAME]] [-o [OUTPUT]]\n\n" \
            "Please use the following command for detailed usage information: \n" \
            "\t> axelf.py --help\n"

class ArgumentParser() :
    def __init__(self) :
        self.parser = argparse.ArgumentParser(
                prog="axelf.py",
                formatter_class=argparse.RawDescriptionHelpFormatter,
                usage="" \
                    "axelf.py [-h] [-p] inputFile -s [TPM_SOURCE] [-a " \
                    "[ALLELE_LIST]] [-c [CANCER_TYPE]] [-l [PEPTIDE_LENGTH]] " \
                    "[-g [GENE_NAME]] [-o [OUTPUT]]",
                description='''\
                Utilizing biophysical principles and existing MHC binding 
                prediction tools in concert with experimental RNA expression 
                values, AXEL-F derive a function that estimates the likelihood 
                of a peptide being presented on a given MHC class I molecule.
                '''
            )

        # Positional Arguments
        self.parser.add_argument('inputFile', 
                            type=argparse.FileType('r'), 
                            nargs='+',
                            # action=SetFileExtension,
                            help='''\
                                FASTA file or CSV file containing peptides with 
                                or without additional data such as alleles, 
                                gene name/id, tpm, etc.
                            '''
                        )

        # Hidden Arguments
        self.parser.add_argument('--inputFileType', 
                            required=False, 
                            type=str,
                            nargs="?", 
                            default="", 
                            help=argparse.SUPPRESS
                        )

        # Optional Arguments
        # TODO : Maybe add 'metavar='
        self.parser.add_argument('-s', '--tpm-source', 
                            required=False,
                            type=str,
                            nargs="?", 
                            choices=["custom", "tcga"], 
                            default="custom", 
                            const="custom",
                            help="Specify where TPM values are coming from."
                        )

        self.parser.add_argument('-t', '--tpm', 
                            type=float,
                            nargs="?", 
                            action=ValidateTPM,
                            help="TPM value."
                        )

        self.parser.add_argument('-g', '--gene-name', 
                            type=str,
                            nargs="?", 
                            action=ValidateGeneName,
                            help="Gene name (e.g. RNF44)."
                        )

        self.parser.add_argument('-c', '--cancer-type', 
                            type=str,
                            nargs="?", 
                            action=ValidateCancerType,
                            help="Acronym of the cancer type (e.g. GBM, LGG)."
                        )

        self.parser.add_argument('-l', '--peptide-length', 
                            type=int,
                            nargs="?", 
                            action=ValidatePeptideLength,
                            help="Path to the file containing length of " \
                                "peptide."
                        )

        self.parser.add_argument('-a', '--allele-list', 
                            type=str,
                            nargs="?", 
                            action=ValidateAlleles,
                            help="Path to the file containing list of alleles."
                        )

        self.parser.add_argument('-o', '--output', 
                            type=argparse.FileType('w', encoding='UTF-8'),
                            nargs="?", 
                            help="Path to the output file."
                        )

        self.parser.add_argument('-p', '--print-alleles', 
                            nargs="?",
                            action=AvailableAlleles,
                            help="Path to the output file."
                        )

        self.parser.add_argument('-d', '--print-cancer-types', 
                            nargs="?",
                            action=AvailableCancerTypes,
                            help="Path to the output file."
                        )

    def parse_arguments(self) :
        return self.parser.parse_args()


class AvailableAlleles(argparse.Action) :
    def __call__(self, *args) :
        print(", ".join(MHCIAlleleData().get_all_allele_names()))
        sys.exit(0)

class AvailableCancerTypes(argparse.Action) :
    def __call__(self, *args) :
        print("Here are list of available cancer types:")
        print(", ".join(list(Axelf().tcga_data.columns)[2:]))
        sys.exit(0)

class ValidateAlleles(argparse.Action) :
    def __call__(self, *args) :
        '''
        Description:
        Checks if user's input allele is found in MHCI alleles.

        Parameters:
        *args - tuple containing (ArgumentParser obj, Namespace obj, user input, flag)
        '''
        NAMESPACE = 1
        namespace = args[NAMESPACE]
        value = args[NAMESPACE+1]
        mhci_alleles = MHCIAlleleData().get_all_allele_names()
        mhci_alleles = [allele.lower() for allele in mhci_alleles]
        
        # TODO: Create command to get all alleles.
        if value.lower() not in mhci_alleles :
            raise ValueError("%s is not valid allele name. Valid allele names " \
                            "will look like the following.\n" \
                            "BoLA-1:00901 for cow alleles, " \
                            "HLA-A*01:01 for human alleles, " \
                            "H-2-Db for mouse alleles, " \
                            "etc.\n" \
                            "To see all the available alleles, use '-p' flag." \
                            "\n\n%s" %(value, USAGE_INFO))
        
        setattr(namespace, self.dest, value)

class ValidatePeptideLength(argparse.Action) :
    def __call__(self, *args) :
        NAMESPACE = 1
        namespace = args[NAMESPACE]
        value = args[NAMESPACE+1]

        if not isinstance(value, int) :
            raise TypeError("%s is invalid peptide length. " \
                            "Peptide length must be an integer." \
                            "\n\n%s" %(value, USAGE_INFO))
        
        if value < 8 or 14 < value :
            raise ValueError("Length must be a value between 8 and 14.")
        
        setattr(namespace, self.dest, value)

class ValidateTPM(argparse.Action) :
    def __call__(self, *args) :
        NAMESPACE = 1
        namespace = args[NAMESPACE]
        value = args[NAMESPACE+1]
        
        if not isinstance(value, float) :
            raise TypeError("%s is invalid TPM value. " \
                            "TPM value must be integer or decimal value." \
                            "\n\n%s" %(value, USAGE_INFO))
        
        setattr(namespace, self.dest, value)

class ValidateGeneName(argparse.Action) :
    def __call__(self, *args) :
        NAMESPACE = 1
        namespace = args[NAMESPACE]
        value = args[NAMESPACE+1]
        available_genes = Axelf().tcga_data["gene"].values

        if value not in available_genes :
            raise ValueError("%s is invalid gene name. Please re-enter the correct gene name." \
                            "\n\n%s" %(value, USAGE_INFO))

        setattr(namespace, self.dest, value)

class ValidateCancerType(argparse.Action) :
    def __call__(self, *args) :
        NAMESPACE = 1
        namespace = args[NAMESPACE]
        value = args[NAMESPACE+1]
        available_cancertypes = list(Axelf().tcga_data.columns)[2:]

        if value not in available_cancertypes :
            raise ValueError("%s is invalid cancer. Please re-enter the correct cancer type." \
                            "\n\n%s" %(value, USAGE_INFO))

        setattr(namespace, self.dest, value)