import unittest
import sys, os
from pathlib import Path
PDIR = Path(os.path.abspath(__file__)).parent.parent.absolute()
TEST_DIR = str(PDIR) + "/tests"
AXELF_DIR = str(PDIR) + "/method/axelf-executable"
sys.path.insert(0, AXELF_DIR)
from axelf_executable import Axelf


class TestAxelF(unittest.TestCase):
    validator = Axelf()

    def test_basic_validation(self):
        # Gene Name
        self.assertTrue(self.validator.validate_gene_name("7SK"))
        self.assertTrue(self.validator.validate_gene_name("AAAS"))

        # Ensemble ID
        self.assertTrue(self.validator.validate_ensemble_id("ENSG00000276626"))
        self.assertTrue(self.validator.validate_ensemble_id("ENSG00000157426"))

        # Cancer Type
        self.assertTrue(self.validator.validate_cancer_type("GBM"))
        self.assertTrue(self.validator.validate_cancer_type("CESC"))


    def test_input_csv(self):
        # Custom CSV
        csv_file = "{}/data/input/sample_input.csv".format(TEST_DIR)
        self.validator.validate_csv(csv_file)

        # TCGA
        csv_file = "{}/data/input/sample_input2.csv".format(TEST_DIR)
        self.validator.validate_csv(csv_file, tpm_source="tcga")


    def test_get_tcga(self):
        # Providing Gene Name
        gene_name = "7SK"
        cancer_type = "GBM"
        expected_result = 150.0397
        result = self.validator.get_tcga(gene_name, cancer_type)
        self.assertEqual(expected_result, result)

        gene_name = "AAAS"
        cancer_type = "CESC"
        expected_result = 34.0103
        result = self.validator.get_tcga(gene_name, cancer_type)
        self.assertEqual(expected_result, result)

        # Providing Enemble ID
        ensemble_id = "ENSG00000277313" # 7SK
        cancer_type = "GBM"
        expected_result = 0.0009999891
        result = self.validator.get_tcga(None, cancer_type, ensemble_id=ensemble_id)
        self.assertEqual(expected_result, result)

        ensemble_id = "ENSG00000094914" # AAAS
        cancer_type = "CESC"
        expected_result = 34.0103
        result = self.validator.get_tcga(None, cancer_type, ensemble_id=ensemble_id)
        self.assertEqual(expected_result, result)

        # Providing Both
        gene_name = "7SK"
        ensemble_id = "ENSG00000277313" # 7SK
        cancer_type = "GBM"
        expected_result = 0.0009999891
        result = self.validator.get_tcga(gene_name, cancer_type, ensemble_id=ensemble_id)
        self.assertEqual(expected_result, result)

        gene_name = "AAAS"
        ensemble_id = "ENSG00000094914" # AAAS
        cancer_type = "CESC"
        expected_result = 34.0103
        result = self.validator.get_tcga(gene_name, cancer_type, ensemble_id=ensemble_id)
        self.assertEqual(expected_result, result)

        # Providing Nothing
        self.assertRaises(ValueError, self.validator.get_tcga, None, cancer_type)


if __name__=="__main__":
    unittest.main()