import pkg_resources
import os
import sys
from subprocess import Popen, PIPE
import tempfile
import logging
logger = logging.getLogger(__name__)

netmhc_script = pkg_resources.resource_filename('netmhc_3_4_executable', 'netMHC')

def predict(allele_name, binding_length, peptide, format='fasta'):
    accepted_formats = ['fasta', 'peptide']
    format = format.lower()
    if format not in accepted_formats:
        raise ValueError('format must be one of {}'.format(accepted_formats))

    # Eliminate the asterisk (*) from the allele_name
    allele_name = allele_name.replace('*', '')

    # Make a file with the peptide as expected by netMHC.
    infile = tempfile.NamedTemporaryFile(suffix='-ann-input')
    logger.info('input filename: {}'.format(infile.name))
    infile_contents = ">test\n{}\n".format(peptide)
    logging.debug('infile contents:\n{}'.format(infile_contents))
    infile.write(infile_contents)
    infile.seek(0)

    # Execute netMHC & collect the output as a list of lines.
    # sys.executable is the full path of the currently running python interpreter.
    cmd = [sys.executable, netmhc_script, '-a', allele_name, '-l', str(binding_length), infile.name]
    process = Popen(cmd, stdout=PIPE)
    (stdoutdata, stderrdata) = process.communicate()
    lines = stdoutdata.split('\n')

    if process.returncode != 0:
        msg = "ANNPredictor did not execute.  Full command: {}\n".format(cmd)
        # Path: {}, input: {}, Process ID: {}.".format(executable_path, infile.name, pid)
        raise Exception(msg)

    # The scores returned by netMHC are ic50.
    ic50s = []
    for line in lines:
        row = line.split()
        if 'Entry_1' in row:  # for the list with 'Entry_1' element
            ic50 = float(row[3])
            ic50s.append(ic50)

    infile.close()  # This automatically deletes the input file.
    return tuple(ic50s)

def predict_from_peptide_file(allele_name, binding_length, peptide_file_path):
    """ @brief Performs a prediction with a file containing a different peptide on each line as
            input.
        This is a special mode of netMHC that is used for efficient calculation of score
        distributions used in generating percentiles from raw scores.
        @author: Jivan
        @since: 2015-09-02
    """
    cmd = [
        sys.executable, netmhc_script,
        '-a', allele_name, '-l', str(binding_length), '-p', peptide_file_path,
    ]

    process = Popen(cmd, stdout=PIPE)
    stdoutdata, stderrdata = process.communicate()
    lines = stdoutdata.split('\n')
    scores = []
    for line in lines:
        line_pieces = line.split()
        if 'Entry_1' in line_pieces:
            scores.append(line_pieces[3])

    return scores

allowed_binding_lengths = (8, 9, 10, 11, 12, 13, 14, 15)
# Retreived from http://www.cbs.dtu.dk/services/NetMHC/MHC_allele_names.txt on 2015-09-02
allowed_allele_names = [
    'BoLA-D18.4',
    'BoLA-HD6',
    'BoLA-JSP.1',
    'BoLA-T2C',
    'BoLA-T2a',
    'BoLA-T2b',
    'H-2-Db',
    'H-2-Dd',
    'H-2-Kb',
    'H-2-Kd',
    'H-2-Kk',
    'H-2-Ld',
    'HLA-A01:01',
    'HLA-A02:01',
    'HLA-A02:02',
    'HLA-A02:03',
    'HLA-A02:06',
    'HLA-A02:11',
    'HLA-A02:12',
    'HLA-A02:16',
    'HLA-A02:17',
    'HLA-A02:19',
    'HLA-A02:50',
    'HLA-A03:01',
    'HLA-A11:01',
    'HLA-A23:01',
    'HLA-A24:02',
    'HLA-A24:03',
    'HLA-A25:01',
    'HLA-A26:01',
    'HLA-A26:02',
    'HLA-A26:03',
    'HLA-A29:02',
    'HLA-A30:01',
    'HLA-A30:02',
    'HLA-A31:01',
    'HLA-A32:01',
    'HLA-A32:07',
    'HLA-A32:15',
    'HLA-A33:01',
    'HLA-A66:01',
    'HLA-A68:01',
    'HLA-A68:02',
    'HLA-A68:23',
    'HLA-A69:01',
    'HLA-A80:01',
    'HLA-B07:02',
    'HLA-B08:01',
    'HLA-B08:02',
    'HLA-B08:03',
    'HLA-B14:02',
    'HLA-B15:01',
    'HLA-B15:02',
    'HLA-B15:03',
    'HLA-B15:09',
    'HLA-B15:17',
    'HLA-B18:01',
    'HLA-B27:05',
    'HLA-B27:20',
    'HLA-B35:01',
    'HLA-B35:03',
    'HLA-B38:01',
    'HLA-B39:01',
    'HLA-B40:01',
    'HLA-B40:02',
    'HLA-B40:13',
    'HLA-B42:01',
    'HLA-B44:02',
    'HLA-B44:03',
    'HLA-B45:01',
    'HLA-B46:01',
    'HLA-B48:01',
    'HLA-B51:01',
    'HLA-B53:01',
    'HLA-B54:01',
    'HLA-B57:01',
    'HLA-B58:01',
    'HLA-B73:01',
    'HLA-B83:01',
    'HLA-C03:03',
    'HLA-C04:01',
    'HLA-C05:01',
    'HLA-C06:02',
    'HLA-C07:01',
    'HLA-C07:02',
    'HLA-C08:02',
    'HLA-C12:03',
    'HLA-C14:02',
    'HLA-C15:02',
    'HLA-E01:01',
    'Mamu-A01',
    'Mamu-A02',
    'Mamu-A07',
    'Mamu-A11',
    'Mamu-A20102',
    'Mamu-A2201',
    'Mamu-A2601',
    'Mamu-A70103',
    'Mamu-B01',
    'Mamu-B03',
    'Mamu-B08',
    'Mamu-B1001',
    'Mamu-B17',
    'Mamu-B3901',
    'Mamu-B52',
    'Mamu-B6601',
    'Mamu-B8301',
    'Mamu-B8701',
    'Patr-A0101',
    'Patr-A0301',
    'Patr-A0401',
    'Patr-A0701',
    'Patr-A0901',
    'Patr-B0101',
    'Patr-B1301',
    'Patr-B2401',
    'SLA-10401',
    'SLA-20401',
    'SLA-30401',
]
