#!/usr/bin/env python
from __future__ import print_function
from bisect import bisect, bisect_left
import os
import traceback, time
import math
import configparser
import sys
import re
import csv
import pickle
import logging
logging.basicConfig(level=logging.WARNING, format='%(asctime)s,%(msecs)d %(levelname)-8s [%(filename)s:%(lineno)d] %(message)s', datefmt='%Y-%m-%d:%H:%M:%S',)

from length_rescaling import calculate_length_rescaled

# adding all methods to the python path
script_dir = os.path.dirname(os.path.realpath(__file__))
methods_dir = script_dir + '/methods'
methods_base_dirs = (
    'allele-info', 
    'iedbtools-utilities', 
    'mhcii-tepitope-predictor',
    'mhcii-comblib-predictor',
    'mhcii-comblib-percentile-data',
    #'mhcii-netmhciipan-percentile-data',
    #'mhcii-nnalign-percentile-data',
    'mhcii-smmalign-percentile-data',
    'mhcii-tepitope-percentile-data',
    'mhcii-netmhciipan-3.2-percentile-data',
    'mhcii-netmhcii-2.3-percentile-data',
    'mhcii-predictor-data',
    #'netmhciipan-3.1-executable',
    'netmhciipan-3.2-executable',
    'netmhcii-1.1-executable',
    #'netmhcii-2.2-executable',
    'netmhcii-2.3-executable',
    'netmhciipan-4.1-executable',
    'mhcii-netmhciipan-4.1-ba-percentile-data',
    'mhcii-netmhciipan-4.1-el-percentile-data',
)
for method_base_dir in methods_base_dirs:
    sys.path.append(methods_dir + '/' + method_base_dir)

from mhcii_predictor_data import get_method_allele_list

from allele_info import MHCIIAlleleData

from mhcii_predictor import MhciiPredictor
from common_bio import Proteins

from print_allele import get_available_alleles

def eprint(*args, **kwargs):
    print(*args, file=sys.stderr, **kwargs)


class InputError(Exception):
    """Exception raised for errors in the input."""

    def __init__(self, value):
            self.value = value

    def __str__(self):
            return self.value

class UnexpectedInputError(Exception):
    """Exception raised for errors in the input."""

    def __init__(self, value):
            self.value = value

    def __str__(self):
            return self.value


class PredictionTable:
    """Generates a table from a set of sequences and predictions"""
    def __init__(self, form):
        self.processing_column={"Proteasome_score":4, "TAP_score":5, "MHC_score":6, "Processing_score":7, "Total_score":8}
        self.sort_output = form["sort_output"]
        self.row_data = []

    def extractCutoff(self, form):
        if form["cutoff_type"] == "none":
            self.use_cutoff=False
        else:
            self.use_cutoff=True
            try:
                self.cutoff_value = float(form["cutoff_value"])
            except:
                raise InputError("Invalid entry for output cutoff <br/>Please enter a numeric value.")
            if form["cutoff_type"] != "MHC_IC50":
                try:
                    self.cutoff_column = self.processing_column[form["cutoff_type"]]
                except:
                    raise UnexpectedInputError("Invalid cutoff option: " + form["cutoff_type"])

    def sortRows(self):
        if self.sort_output == "MHC_IC50":
            if form["pred_method"] in ["Sturniolo", "netmhciipan_el"]:
                self.row_data.sort(lambda x,y: cmp(x[-1], y[-1]))
                self.row_data.reverse()
            else:
                self.row_data.sort(lambda x,y: cmp(x[-1], y[-1]))
        elif self.sort_output =="position_in_sequence":
            pass
        else:
            try:
                col = self.processing_column[self.sort_output]
            except:
                raise UnexpectedInputError("Invalid sorting option: " + self.sort_output)
            self.row_data.sort(lambda x,y: -cmp(x[col], y[col]))

    def add_rows_binding(self, allele, pep_length, proteins, scores):
        logging.debug((allele, pep_length, proteins, scores))
        if form["pred_method"] == "consensus3" or form["pred_method"] == "IEDB_recommended":
            for (i,(sequence, predictions)) in enumerate(zip(proteins.sequences,scores)):
                dummy = []
                for (k, (dummy)) in enumerate(predictions):
                    peptide_sequence = sequence[k : k + pep_length]
                    try:
                        temp_index = peptide_sequence.index(peptide_sequence)
                    except:
                        raise ValueError("Core sequence and peptide sequence doesn't match.")
                    dummy_list = []
                    dummy_list.append(allele)
                    # sequence number
                    dummy_list.append(i+1)
                    # peptide start position
                    dummy_list.append(k+1)
                    # peptide end position
                    dummy_list.append(k+pep_length)
                    # peptide length
                    dummy_list.append(str(pep_length))                    
                    dummy_list.append(peptide_sequence)
                    for item in dummy:
                        dummy_list.append(item)
                    self.row_data.append(tuple(dummy_list))
                    
        else:
            for (i,(sequence, predictions)) in enumerate(zip(proteins.sequences,scores)):
                for (k, (core, prediction, rank)) in enumerate(predictions):
                    peptide_sequence = sequence[k : k + pep_length]
                    core_sequence = core
                    #try:
                    logging.debug(peptide_sequence)
                    logging.debug(core_sequence)
                    logging.debug(pep_length)
                    #temp_index = peptide_sequence.index(core_sequence)
                    #except:
                    #    raise ValueError("Core sequence and peptide sequence doesn't match.")
                    peptide_source = i+1
                    peptide_start = k + 1
                    peptide_end = k + pep_length
                    self.row_data.append((allele, peptide_source, peptide_start, peptide_end, str(pep_length), core_sequence, peptide_sequence, prediction, rank))

    def format_binding(self, proteins, results):
        for(pep_length, allele, scores) in results:
            self.add_rows_binding(allele,pep_length, proteins, scores)
        if form["pred_method"] not in ['netmhciipan_el', 'netmhciipan_ba']:
            self.row_data = add_column_adj_rank(self.row_data, form["pred_method"])
        return self.row_data

    def add_rows_processing(self, allele, pep_length, proteins, mhc_scores, tap_scores, prot_scores):
        logging.debug((allele, pep_length, proteins, mhc_scores, tap_scores, prot_scores))
        prot_offset = pep_length-6
        for (i,(sequence, mhc, tap, prot)) in enumerate(zip(proteins.sequences,mhc_scores, tap_scores, prot_scores)):
            for nterm in range(len(sequence)-(pep_length-1)):
                mhc_ic50  = mhc[nterm]
                mhc_score = -math.log10(mhc_ic50)
                tap_score = tap[nterm]
                prot_score = prot[nterm+prot_offset]
                proc_score = prot_score + tap_score
                total_score = mhc_score+tap_score+ prot_score
                sequence_position = "%s:%s-%s" %(i+1, nterm + 1, nterm + int(pep_length))
                scanned_sequence = sequence[nterm : nterm + pep_length]
                row = (allele, sequence_position, pep_length, scanned_sequence, prot_score, tap_score, mhc_score, proc_score, total_score, mhc_ic50)
                self.row_data.append(row)

    def format_processing(self, proteins, mhc_results, tap_results, prot_scores):
        for (pep_length, allele, mhc_scores) in mhc_results:
            for (pep_length2, tap_scores) in tap_results:
                if pep_length==pep_length2:
                    self.add_rows_processing(allele, pep_length, proteins, mhc_scores, tap_scores, prot_scores)
                    break
        self.sortRows()
        return self.row_data

    def html_table_input_sequences(self, proteins):
        result="""
        <table id='input_table' class="result" border="2" >
            <thead><tr><th>#</th><th>Name</th><th>Sequence</th></thead>
            <tbody>
        """
        for i,(name, sequence) in enumerate(zip(proteins.names, proteins.sequences)):
            if i % 2==0:
                rowstyle = 'class="even_row"'
            else:
                rowstyle = ""
            blocklen = 50
            if len(sequence)>blocklen:
                seqblock=""
                for block in range(0,len(sequence)-blocklen,blocklen):
                    seqblock+="""%s<br/>""" %sequence[block:block+blocklen]
                seqblock+=sequence[block+blocklen:]
                result+='<tr %s><td>%d</td><td>%s</td><td class ="sequence">%s</td></tr>' % (rowstyle, i+1,name, seqblock)
            else:
                result+='<tr %s><td>%d</td><td>%s</td><td class = "sequence">%s</td></tr>' % (rowstyle, i+1,name, sequence)
        result+="</tbody></table>"
        return result




def print_input_page(sequence):
        config_parser = configparser.ConfigParser()
        config_parser.read("../setup.cfg")
        html_path=config_parser.get("path", "html")
        template = open(html_path + "/html/mhc_II_binding.seq","r").read()
        print( "Content-Type: text/html")
        print( "")
        print( template % sequence)


def sort_table(method, table_rows):
    if not table_rows:
        return table_rows
    method = method.lower()
    if not method in ['consensus3', 'iedb_recommended', ]:
        table_rows = sorted(table_rows, key=lambda tup: float(tup[-1]))
    if method.lower() in ['consensus3', 'iedb_recommended', ]:
        table_rows = sorted(table_rows, key=lambda tup: float(tup[7]))
    elif method in ['sturniolo', ]:
        table_rows = sorted(table_rows, key=lambda tup: float(tup[7]), reverse=True)
        table_rows = sorted(table_rows, key=lambda tup: float(tup[9]))
    elif method in ['netmhciipan_el', ]:
        table_rows = sorted(table_rows, key=lambda tup: float(tup[7]), reverse=True)
        table_rows = sorted(table_rows, key=lambda tup: float(tup[8]))
    elif method in ['netmhciipan_ba', ]:
        table_rows = sorted(table_rows, key=lambda tup: float(tup[7]))
        table_rows = sorted(table_rows, key=lambda tup: float(tup[8]))
    else:
        table_rows = sorted(table_rows, key=lambda tup: float(tup[7]))
        table_rows = sorted(table_rows, key=lambda tup: float(tup[9])) 
    return table_rows

def form_valid(form):

    method = form["pred_method"]
    form_alleles = form["allele"]
    form_alleles = form_alleles.replace("HLA-", "")       
    #Validate method selection
    method_list = {
            "consensus3":'consensus',
            "IEDB_recommended": 'netmhciipan_el',
            "NetMHCIIpan":'NetMHCIIpan',
            "nn_align":"nn_align-2.3",
            "smm_align":"smm_align",
            "sturniolo":'tepitope',
            "comblib":"comblib",
            "netmhciipan_el":'netmhciipan_el',
            "netmhciipan_ba":'netmhciipan_ba',
        }
    if method not in method_list:
        raise UnexpectedInputError("Selected prediction method '%s' does not exist. To see all available methods, please type command:\n$ python mhc_II_binding.py method\n" % method)
    else:
        method = method_list[method]
            
    allele_list = form_alleles.split(',')  
    length = form['length']
    length_list = length if ( type(length) is list) else  length.split(',')
    length_list = [item.strip() for item in length_list]
    # get_min_length
    min_length = min([int(i.split('-')[0]) for i in length_list])

    miiad = MHCIIAlleleData()
    available_allele_names = miiad.get_allele_names(method_name = method)
    Invalid_alleles = [allele for allele in allele_list if allele not in available_allele_names]
    if Invalid_alleles:
        raise UnexpectedInputError('Invalid allele name "%s" for method "%s" was found. To see all available alleles, please type command:\n$ python mhc_II_binding.py allele' % (', '.join(Invalid_alleles), form["pred_method"]))
    proteins=Proteins(form["sequences"])
    for sequence in proteins.sequences:
        if len(sequence.strip())<11:
            raise UnexpectedInputError('The length of the input protein sequence must be at least 11, please check your sequence: "%s".' % sequence)
        elif len(sequence.strip())<min_length:
            raise UnexpectedInputError('The length of the input protein sequence should be no less than the input length, please check your sequence: "%s".' % sequence)
    # Check if string is DNA sequence
    for protein_sequence in proteins.sequences:
        if re.match('^[ACGT]+$', protein_sequence.upper()):                 
            return 'DNA_sequence_input'
    return True
   
def get_allele_length_combo_list(allele_list, length_list):
    # if one of the lengths is 'asis', all have to be 'asis'
    if 'asis' in length_list or 'as_is' in length_list:
        length_list = ['asis',]*len(allele_list)
        return list(zip(allele_list, length_list))

    allele_length_combo_list = []
    for lengths in length_list:
        # lengths should be like "15" or "12-18" or "10,12-18,20" or "asis"   
        lengths = str(lengths).strip()
        if not re.match(r'^[\d,\s-]+$', lengths):
            raise ValueError('invalid length input: %s' % lengths)
        # length should be like "15" or "12-18"
        for length in lengths.split(','):
            length = length.strip()
            if "-" in length:
                if length.count("-") > 1:
                    raise ValueError('invalid length input: %s' % length)
                len_start, len_end = [l.strip() for l in length.split("-")]
                if not (len_start.isdigit() and len_end.isdigit()):
                    raise ValueError('invalid length input: %s' % length)
                for length in range(int(len_start),int(len_end)+1):
                    for allele in allele_list:
                        allele_length_combo_list.append((allele,length))
            else:
                if not length.isdigit():
                    raise ValueError('invalid length input: %s' % length)
                length = int(length)
                for allele in allele_list:
                    allele_length_combo_list.append((allele,length))
    allele_length_combo_list = list(set(allele_length_combo_list))
    allele_length_combo_list.sort()
    return allele_length_combo_list
 
def add_column_adj_rank(test_data, method=None):
    logging.debug( 'add_column_adj_rank for method: %s' % method)
    result_list = []
    method = method.lower()
    for row in test_data:

        if method == 'iedb_recommended':
            logging.debug('method == iedb_recommended')
            logging.debug(repr(row))
            row = list(row)
            #            header = ('allele', 'seq_num', 'start', 'end', 'length', 'method', 'peptide',  'percentile_rank', unit, 'comblib_core', 'comblib_score', 'comblib_rank','comblib_adjusted_rank', 'smm_align_core', 'smm_align_ic50', 'smm_align_rank', 'smm_align_adjusted_rank', 'nn_align_core', 'nn_align_ic50', 'nn_align_rank', 'nn_align_adjusted_rank', 'netmhciipan_core', 'netmhciipan_ic50', 'netmhciipan_rank', 'netmhciipan_adjusted_rank', 'sturniolo_core', 'sturniolo_score', 'sturniolo_rank', 'sturniolo_adjusted_rank')            
            for i in [21,  18,  15,  12,  9,  6]:
                # for each rank
                row.insert(i+1, calculate_length_rescaled(length=row[4],rank=row[i]))
            logging.debug(repr(row))
            result_list.append(tuple(row))
        elif method == 'consensus3':
            row = list(row)
            #header = ('allele', 'seq_num', 'start', 'end', 'length', 'peptide',  'percentile_rank', unit, 'comblib_core', 'comblib_score', 'comblib_rank', 'comblib_adjusted_rank', 'smm_align_core', 'smm_align_ic50', 'smm_align_rank', 'smm_align_adjusted_rank', 'nn_align_core', 'nn_align_ic50', 'nn_align_rank', 'nn_align_adjusted_rank')
            for i in [18, 15, 12, 9, 6]:
                # for each rank
                row.insert(i+1, calculate_length_rescaled(length=row[4],rank=row[i]))
            result_list.append(tuple(row))
        else:
            #logging.debug('row:%s' % row)
            row = list(row)
            row.append(calculate_length_rescaled(length=row[4],rank=row[-1]))
            result_list.append(tuple(row))
    return result_list

def main(form):

    if "sequence" in form:
        print_input_page(form["sequence"])
    elif "sequence_format" not in form:
        print_input_page("")
    else:
        try:
            input_params = form_valid(form)
            if not input_params:
                return

            proteins=Proteins(form["sequences"])

            method_dict = {
                "consensus3":'consensus',
                "IEDB_recommended": 'netmhciipan_el',
                "NetMHCIIpan":'NetMHCIIpan',
                "nn_align":"nn_align",
                "comblib":"comblib",
                "smm_align":"smm_align",
                "sturniolo":'tepitope',
                "netmhciipan_el":'netmhciipan_el',
                "netmhciipan_ba":'netmhciipan_ba',
            }
            method = method_dict[form["pred_method"]]

            
            alleles = form["allele"]
            alleles = alleles.replace('HLA-','').replace("/","-")  
            allele_list = alleles.split(',')  
            length = form['length']
            length_list = length if ( type(length) is list) else  length.split(',')
            length_list = [item.strip() for item in length_list]


            allele_length_combo_list = get_allele_length_combo_list(allele_list, length_list)
            allele_list,length_list = zip(*allele_length_combo_list)

            pre=MhciiPredictor(method, allele_list, length_list)      
                              
            mhc_scores = pre.predict(proteins.sequences)
             

            table = PredictionTable(form)
            table_rows = table.format_binding(proteins, mhc_scores)
            table_rows = sort_table(form["pred_method"], table_rows)
        
            con_status = {}
            con_list = get_method_allele_list('consensus').strip().split("\n")
            for con_element in con_list:
                con_arr = con_element.split("\t")
                con_status[con_arr[0]] = con_arr[1:]
        except Exception as inst:
            sys.exit(inst)
        else:
            if form["pred_method"] == "consensus3":
                print( "allele\tseq_num\tstart\tend\tlength\tpeptide\tconsensus_percentile_rank\tadjusted_consensus_percentile_rank\tcomblib_core\tcomblib_score\tcomblib_rank\tadjusted_comblib_rank\tsmm_align_core\tsmm_align_ic50\tsmm_align_rank\tadjusted_smm_align_rank\tnn_align_core\tnn_align_ic50\tnn_align_rank\tadjusted_nn_align_rank\tsturniolo_core\tsturniolo_score\tsturniolo_rank\tadjusted_sturniolo_rank")
            elif form["pred_method"] == "IEDB_recommended":
                print( "allele\tseq_num\tstart\tend\tlength\tmethod\tpeptide\tconsensus_percentile_rank\tadjusted_consensus_percentile_rank\tcomblib_core\tcomblib_score\tcomblib_rank\tadjusted_comblib_rank\tsmm_align_core\tsmm_align_ic50\tsmm_align_rank\tadjusted_smm_align_rank\tnn_align_core\tnn_align_ic50\tnn_align_rank\tadjusted_nn_align_rank\tnetmhciipan_core\tnetmhciipan_ic50\tnetmhciipan_rank\tadjusted_netmhciipan_rank\tsturniolo_core\tsturniolo_score\tsturniolo_rank\tadjusted_sturniolo_rank")
            elif form["pred_method"] in ["sturniolo", ]:
                print( "allele\tseq_num\tstart\tend\tlength\tcore_peptide\tpeptide\tscore\tpercentile_rank\tadjusted_rank")
            elif form["pred_method"] in ["netmhciipan_el"]:
                print( "allele\tseq_num\tstart\tend\tlength\tcore_peptide\tpeptide\tscore\tpercentile_rank")
            elif form["pred_method"] in ["netmhciipan_ba"]:
                print( "allele\tseq_num\tstart\tend\tlength\tcore_peptide\tpeptide\tic50\tpercentile_rank")
            else:
                print( "allele\tseq_num\tstart\tend\tlength\tcore_peptide\tpeptide\tic50\tpercentile_rank\tadjusted_rank")
            
            for table_row in table_rows:
                table_row = list(table_row)
                
                if 'H2' not in table_row[0]:
                    table_row[0] = "HLA-%s" % table_row[0].replace("-", "/")
                logging.debug(form["pred_method"])    
                if form["pred_method"] == "consensus3":
                    print ('\t'.join(map(str, tuple(table_row))))
                elif form["pred_method"] == "IEDB_recommended":
                    method_used = table_row[-1].upper().replace("STURNIOLO","Sturniolo").replace('NETMHCIIPAN','NetMHCIIpan')
                    method_list = method_used.split(',')
                    if len(method_list) <= 1:  table_row.insert(5,method_used)
                    else:
                        method_used = "Consensus("+method_used+")"
                        table_row.insert(5,method_used)
                    logging.debug(len(table_row))
                    print( '\t'.join(map(str, tuple(table_row[:-1]))))
                else:
                    # 6 decimal for netmhciipan_el scores
                    if form["pred_method"] == "netmhciipan_el":
                        table_row[-2] = '%.6f' % float(table_row[-2])
                    # 2 decimal for ic50/scores: 1.607724271704614  ->  1.61
                    elif form["pred_method"] == "netmhciipan_ba":
                        table_row[-2] = '%.2f' % float(table_row[-2])
                    else:
                        table_row[-3] = '%.2f' % float(table_row[-3])
                    print("\t".join(map(str, tuple(table_row))))

            if input_params == 'DNA_sequence_input':
                eprint ('# Warning: Potential DNA sequence(s) found! This tool is intended to predict for amino acid sequences. Please double check your input fasta file.')

def commandline_help():
    print( " ________________________________________________________________________________________")
    print( "|****************************************************************************************|")
    print ("| * List all available commands.                                                         |")
    print( "| python mhc_II_binding.py                                                               |")
    print ("|________________________________________________________________________________________|")
    print( "| * List all available mhc_II prediction methods.                                        |")
    print ("| python mhc_II_binding.py method                                                        |")
    print( "|________________________________________________________________________________________|")
    print( "| * List all alleles.                                                                    |")
    print( "| python mhc_II_binding.py allele                                                        |")
    print( "|________________________________________________________________________________________|")
    print( "| * Make predictions given a file containing a list of sequences.                        |")
    print( "| python mhc_II_binding.py prediction_method_name allele_name input_sequence_file_name   |")
    print ("| Example: python mhc_II_binding.py consensus3 HLA-DRB1*03:01 test.fasta                 |")
    print ("|________________________________________________________________________________________|")
    print( "| * You may also redirect (pipe) the input file into the script.                         |")
    print( "| Example: echo -e test.fasta | python mhc_II_binding.py consensus3 HLA-DRB1*03:01       |")
    print( "|________________________________________________________________________________________|")

def commandline_method():
    '''Return all available prediction methods.'''
    print( )
    print ("Available methods are:")
    print ("----------------------")
    print( "comblib")
    print ("consensus3")
    print ("netmhciipan_el (IEDB_recommended)")
    print ("netmhciipan_ba")
    print ("nn_align")
    print ("smm_align")
    print ("sturniolo")
    print()
 

def commandline_allele():
    '''Return all available alleles.'''
    looking_up_methods = ['consensus3', 'comblib', 'smm_align', 'nn_align', 'sturniolo']
    allele_info_for_printing = get_available_alleles(looking_up_methods)
    print(allele_info_for_printing)

if __name__ == '__main__':
    import select
    debug = False
    
    method = allele = infile = None
    
    if (len(sys.argv) == 1):
        commandline_help()
    elif ((len(sys.argv) == 2) and (sys.argv[1] == "method")):
        commandline_method()
    elif ((len(sys.argv) == 2) and (sys.argv[1] == "allele")):
        commandline_allele()
    else:
        method = sys.argv[1]
        if len(sys.argv) > 2:
            allele = sys.argv[2]
    
    # If there's input ready, do something, else do something
    # else. Note timeout is zero so select won't block at all.
    if not sys.stdin.isatty():
        stdin = sys.stdin.readline().strip()
        sys.argv.append(stdin)
    sys.argv = list(filter(None,sys.argv))

    if len(sys.argv) > 3:
        infile = sys.argv[3]

    if len(sys.argv) > 4:
        length = sys.argv[4]
    else:
        length = '15'
            
    if not method or not allele or not infile:
        eprint( "# To run the predction, you must specify method, allele and input file name.")
        exit(0)
    
    allele = allele.replace("_","-").replace("H-2","H2")
    with open(infile, 'r') as r_file:
        sequences = r_file.read()

    # for method NetMHCII
    if 'netmhciipan' not in method.lower():
        method = method.replace('netmhcii-2.3','nn_align').replace('netmhcii-1.1','smm_align').replace('netmhcii','nn_align').replace('IEDB_recommended','netmhciipan_el')   
    seq = [('sequence_format', 'auto'), ('sort_output', 'position_in_sequence'), ('cutoff_type', 'none'), ('allele', allele), ('sequence_file', infile), ('pred_method', method), ('sequences', sequences), ('length', length),]
 

   
    form = dict(seq)
    if not debug:
        main(form)
    else:
        calltime = str(time.ctime(time.time()))
        cgilog = open("logs/tool_log.txt","a")
        cgilog.write('\n--- '+ calltime +' ---\n')
        cgilog.write('mhc_II_binding.py\n')
        cgilog.flush()
        cgilog.write('Form keys: %d\n' % len(form.keys()))
        for key in form.keys():
            cgilog.write('%s - "%s" - "%s"\n' % (calltime,key, form[key]))
        cgilog.close()

        try:
            main(form)
        except Exception as inst:
            cgilog = open("logs/tool_log.txt","a")
            cgilog.write('XXX '+ calltime +' XXX\n')
            cgilog.write("EXCEPTION: '%s'\n" % str(inst))
            traceback.print_exc(None, cgilog)
            cgilog.close()
        else:
            cgilog = open("logs/tool_log.txt","a")
            cgilog.write('... '+ calltime +' ...\n')
            cgilog.close()
            
